<?php

namespace Ktane ;

use Exception ;

/**
 * Implémentation du module _Code Morse_ tel que spécifié
 * dans le "Manuel de désamorçage de bombe".
 */
class MorseCode
{
    /**
     * Version du manuel de désamorçage de référence.
     *
     * @return "fr-750"
     */
    public static function version()
    {
        return "fr-750" ;
    }

    /**
     * Configurations possibles d'après le manuel du
     * démineur, version française n°750.
     *
     * Pour éviter les problèmes d'arrondis des flottants,
     * les fréquences sont en KHz et non en MHz.
     */
    private const SETUP = [
        [ "vitre", 3505 ],
        [ "ville", 3515 ],
        [ "chose", 3522 ],
        [ "signe", 3532 ],
        [ "linge", 3535 ],
        [ "ligne", 3542 ],
        [ "champ", 3545 ],
        [ "litre", 3552 ],
        [ "phase", 3555 ],
        [ "chaud", 3565 ],
        [ "bille", 3572 ],
        [ "balle", 3575 ],
        [ "singe", 3582 ],
        [ "plume", 3592 ],
        [ "pluie", 3595 ],
        [ "salle", 3600 ],
    ] ;

    /**
     * Code morse pour les lettres et les chiffres.
     *
     * * Un `.` (point) représente un clignotement court,
     * * Un `-` (tiret) représente un clignotement long,
     * * Un ` ` (espace) représente une pause (fin de lettre).
     */
    private const CODE = [
        "a" => ".- ",
        "b" => "-... ",
        "c" => "-.-. ",
        "d" => "-.. ",
        "e" => ". ",
        "f" => "..-. ",
        "g" => "--. ",
        "h" => ".... ",
        "i" => ".. ",
        "j" => ".--- ",
        "k" => "-.- ",
        "l" => ".-.. ",
        "m" => "-- ",
        "n" => "-. ",
        "o" => "--- ",
        "p" => ".--. ",
        "q" => "--.- ",
        "r" => ".-. ",
        "s" => "... ",
        "t" => "- ",
        "u" => "..- ",
        "v" => "...- ",
        "w" => ".-- ",
        "x" => "-..- ",
        "y" => "-.-- ",
        "z" => "--.. ",
        "1" => ".---- ",
        "2" => "..--- ",
        "3" => "...-- ",
        "4" => "....- ",
        "5" => "..... ",
        "6" => "-.... ",
        "7" => "--... ",
        "8" => "---.. ",
        "9" => "----. ",
    ] ;

    /**
     * La bombe sur laquelle ce module est branché
     */
    private $bomb ;

    /**
     * La configuration choisie
     */
    private $setup ;

    /**
     * Le module est-t-il désactivé ?
     */
    private $disarmed ;

    /**
     * Initialise un Contrôleur morse
     *
     * @param $bomb  La bombe branchée sur ce module
     * @param $setup L'état de configuration du module
     *
     * @throw Exception Si l'état n'est pas dans la plage acceptable
     */
    public function __construct(Bomb $bomb, $setup)
    {
        if (! array_key_exists($setup, self::SETUP)) {
            throw new Exception("State ($setup) is not acceptable") ;
        }
        $this->bomb     = $bomb ;
        $this->setup    = $setup ;
        $this->disarmed = false ;
    }

    /**
     * Indication de désarmement du module.
     *
     * @return Vrai si le module est désarmé, Faux sinon.
     */
    public function disarmed()
    {
        return $this->disarmed ;
    }

    /**
     * Séquence d'allumage de la lumière clignotante.
     *
     * * Un `.` (point) représente un clignotement court,
     * * Un `-` (tiret) représente un clignotement long,
     * * Un ` ` (espace) représente une pause (fin de lettre),
     * * Un `\n` (retour à la ligne) représente une longue pause (fin de mot).
     *
     * @return La chaîne de caractère représentant la séquence de clignotement.
     */
    public function getLightSignal()
    {
        return rtrim(strtr(self::SETUP[$this->setup][0], self::CODE)) . "\n" ;
    }

    /**
     * Lancer la transmission sur la fréquence spécifiée.
     *
     * * Si le module est déjà désarmé, rien ne se passe.
     * * Si la fréquence est la bonne, le module est désarmé (et la bombe
     *   est notifiée de ce changement d'état.
     * * Sinon la bombe est notifiée d'une mauvaise tentative.
     */
    public function transmit($frequency)
    {

        if ($this->disarmed()) {
            return ;
        }

        if ($frequency != self::SETUP[$this->setup][1]) {
            $this->bomb->boom() ;
            return ;
        }

        $this->disarmed = true ;
        $this->bomb->disarmed() ;
    }
}
